# MainWindow.py
from ui import MainWindowUI as UI
from PyQt5.QtWidgets import QFileDialog,QMainWindow
from PyQt5.QtGui import QPixmap, QImage
import cv2
import numpy as np
import easyocr


class MainWindow(QMainWindow,UI):
    def __init__(self):
        super(MainWindow, self).__init__()
        self.setupUi(self)

        self.btn_load.clicked.connect(self.load_image)
        self.btn_detect.clicked.connect(self.detect_license_plate)
        self.btn_plate.clicked.connect(self.get_plate)
        self.btn_seg.clicked.connect(self.seg_char)
        self.btn_reg.clicked.connect(self.reg_plate)
        self.btn_green.clicked.connect(self.detect_green_plate)
    
    def load_image(self):
        options = QFileDialog.Options()
        fileName, _ = QFileDialog.getOpenFileName(self, "QFileDialog.getOpenFileName()", "", "Image Files (*.png *.jpg *.jpeg)", options=options)
        if fileName:
            self.image = cv2.imread(fileName)
            self.display_image(self.image)
    
    def display_image(self,img):
        if len(img.shape)==3:#彩色图像
            height, width, channel = img.shape
            bytes_per_line = channel * width
            q_img = QImage(img.data, width, height, bytes_per_line, QImage.Format_BGR888)
        else:               # 灰度图像或二值图像
            height, width= img.shape
            bytes_per_line = width
            q_img = QImage(img.data, width, height, bytes_per_line, QImage.Format_Grayscale8)            

        self.label.setPixmap(QPixmap.fromImage(q_img))


    #灰度拉伸
    def grayScaleStretch(self,img):
        maxGray = float(img.max())
        minGray = float(img.min())
        for i in range(img.shape[0]):
            for j in range(img.shape[1]):
                img[i,j] = 255 / (maxGray - minGray) * (img[i,j] - minGray)
        return img
 
    #图像二值化
    def image2Binary(self,img):
        #选取灰度最大最小值的中间值
        maxGray = float(img.max())
        minGray = float(img.min())
        threshold = (minGray + maxGray) / 2
        ret,bin = cv2.threshold(img, threshold, 255, cv2.THRESH_BINARY)
        return bin
     
    #图像预处理
    def imagePreProcess(self,img):

        imgGray = cv2.cvtColor(img, cv2.COLOR_BGR2GRAY)
        #灰度拉伸
        imgGray = self.grayScaleStretch(imgGray)
        imgGray = cv2.GaussianBlur(imgGray, (3,3), 5)
        #进行边缘检测
        cannyEdges = cv2.Canny(imgGray, 180, 230)
        #二值化
        imgBinary = self.image2Binary(cannyEdges)
        #plt.imshow(imgBinary, cmap='gray')
        #先做闭运算再做开运算
        kernel = np.ones((3,3), np.uint8)
        
            
        imgOut = cv2.morphologyEx(imgBinary, cv2.MORPH_CLOSE, kernel)
        # cv2.imshow("img",imgOut)
        # cv2.waitKey()
        imgOut = cv2.morphologyEx(imgOut, cv2.MORPH_OPEN, kernel)
        # cv2.imshow("img",imgOut)
        # cv2.waitKey()
        imgOut = cv2.absdiff(imgBinary, imgOut)
        imgOut = cv2.morphologyEx(imgOut, cv2.MORPH_CLOSE, kernel)
        imgOut = cv2.dilate(imgOut, kernel, iterations=1)
        # plt.imshow(imgOut, cmap='gray')
        return imgOut

    #定位车牌
    def locate_plate(self,imgProcessing,img):
        #过滤矩形的参数
        minRectW = 100
        minRectH = 50
        
        contours,hierarchy = cv2.findContours(imgProcessing, cv2.RETR_EXTERNAL, cv2.CHAIN_APPROX_SIMPLE)
        carPlateCandidates = []
        for contour in contours:
            (x,y,w,h) = cv2.boundingRect(contour)
            #过滤掉一些小的矩形
            if (w < minRectW or h < minRectH):
                continue
            #cv2.rectangle(imgOriginal, (int(x), int(y)), (int(x + w),int(y + h)), (0,255,0), 2)
            carPlateCandidates.append([int(x),int(y),int(x + w),int(y + h)])
        
        #plt.imshow(imgOriginal[:,:,::-1])
        maxMean = 0
        target = []
        target_mask = []
        #依次检查候选车牌列表，用HSV颜色空间判别是否是车牌
        for candidate in carPlateCandidates:
            (x0,y0,x1,y1) = candidate
            candidateROI = img[y0:y1,x0:x1]
            hsvROI = cv2.cvtColor(candidateROI, cv2.COLOR_BGR2HSV)
            mask = cv2.inRange(hsvROI, self.hsvLower, self.hsvUpper)

            #使用均值找出蓝色最多的区域
            mean = cv2.mean(mask)

            if mean[0] > maxMean:
                maxMean = mean[0]
                target = candidate
                target_mask = mask
                
        #对target的范围进行缩小，找出蓝色刚开始和结束的坐标
        nonZeroPoints = cv2.findNonZero(target_mask)
       
        sortByX = np.sort(nonZeroPoints, axis=0)
        xMin = sortByX[0][0][0]
        xMax = sortByX[-1][0][0]
        print(sortByX)
        sortByY = np.sort(nonZeroPoints, axis=1)
        yMin = sortByY[0][0][1]
        yMax = sortByY[-1][0][1]
        print(sortByY)
     
     
        print("X min:" + str(xMin) + " X max:" + str(xMax) + " Y min:" + str(yMin) + " Y max:" + str(yMax))
        (x0,y0,x1,y1) = target
        print("Original:" + str(x0) + "," + str(y0) + "," + str(x1) + "," + str(y1))
        #target = (x0 + xMin, y0 + yMin, x0 + (xMax - xMin), y0 + yMax - yMin)
        target = [x0 + xMin, y0 + yMin, x0 + xMax, y0 + yMax]

        return target


    
    def detect_license_plate(self):
        # 车牌识别逻辑（示例：使用OpenCV的车牌检测）
        img = self.image.copy()
        
        #一般情况下，蓝色车牌H分量的值通常都在115附近徘徊
        # S分量和V分量因光照不同而差异较大(opencv2中H分量的取值范围是0到179，而不是图像学中的0到360；S分量和V分量的取值范围是到255)
        deltaH = 15
        self.hsvLower = np.array([115 - deltaH,60,60])
        self.hsvUpper = np.array([115 + deltaH,255,255])
        
        #plt.imshow(imgCarPlate[:,:,::-1])
        img4locate = self.imagePreProcess(img)
        target = self.locate_plate(img4locate, img)
        (x0,y0,x1,y1) = target

        cv2.rectangle(img, (x0,y0), (x1,y1), (0,255,0), 2)

        self.display_image(img)  # 更新显示识别后的图像

    def get_plate(self):
        img = self.image.copy()
        #plt.imshow(imgCarPlate[:,:,::-1])
        img4locate = self.imagePreProcess(img)
        (x0,y0,x1,y1) = self.locate_plate(img4locate, img)

        self.plate=img[y0:y1,x0:x1]
        
        cv2.imwrite('E:\\carplate.jpg',self.plate)
        
        height, width, channel = self.plate.shape
        bytes_per_line = channel * width
        q_img = QImage(bytes(self.plate.data), width, height, bytes_per_line, QImage.Format_BGR888)
        self.label2.setPixmap(QPixmap.fromImage(q_img))

    def seg_char(self):
        plate=self.plate.copy()
        # img1 : resize image
        img1 = cv2.resize(plate, (320, 100), interpolation=cv2.INTER_AREA)
        # img2 : gray image
        img2 = cv2.cvtColor(img1, cv2.COLOR_BGR2GRAY)
        # img3 : noise reduction image
        img3 = cv2.bilateralFilter(img2, 11, 17, 17)
        # img4 : texture information image
        img4 = cv2.Canny(img3, 50, 150)
        # img5 : image removal image
        img5 = img4[10:90, 10:310]
        crop_img = img1[10:90, 10:310, :]
        self.crop=crop_img.copy()
        cv2.imwrite('carplate.jpg',crop_img)
    
        contours, hierarchy = cv2.findContours(img5, cv2.RETR_EXTERNAL, cv2.CHAIN_APPROX_SIMPLE)
        candidate = []
        for cnt in contours:
            x, y, w, h = cv2.boundingRect(cnt)
            if w*h < 500:
                continue
            if w * h > 4000:
                continue
            if h < 20:
                continue
            if w > 80:
                continue
            candidate.append([x, (x + w)])
    
        loc = np.zeros(300)
    
        for j in range(len(candidate)):
            x1 = candidate[j][0]
            x2 = candidate[j][1]
            loc[x1:x2] = 1
    
        start = []
        end = []
    
        if loc[0] == 1:
            start.append(0)
    
        for j in range(300-1):
            if loc[j] == 0 and loc[j+1] == 1:
                start.append(j)
            if loc[j] == 1 and loc[j+1] == 0:
                end.append(j)
    
        if loc[299] == 1:
            end.append(299)
    
        self.char_img=[]    
        if len(start) == 7 and len(end) == 7:
             for j in range(0, 7):
                x1 = start[j]
                x2 = end[j]
                y1 = 0
                y2 = 80
                self.char_img.append(crop_img[y1:y2,x1:x2])
                cv2.imwrite(f'{j+1}.jpg',crop_img[y1:y2,x1:x2])
                cv2.rectangle(crop_img, (x1, y1), (x2, y2), (0, 0, 255), 2)

        # for i in range(0,7):
        #     cv2.imshow('caijian', self.char_img[i])
        #     cv2.waitKey(0)
        
        height, width, channel = crop_img.shape
        bytes_per_line = channel * width
        q_img = QImage(bytes(crop_img.data), width, height, bytes_per_line, QImage.Format_BGR888)
        self.label2.setPixmap(QPixmap.fromImage(q_img))
        
    def reg_plate(self):
        reader = easyocr.Reader(['ch_sim', 'en'])  # 这里的'ch_sim'代表简体中文，'en'代表英文
        result = reader.readtext(self.plate)
        cleaned_text = ''.join([char for char in result[0][1] if char != '.'])
  
        self.label3.setText(cleaned_text)
        
    def detect_green_plate(self):
        # 车牌识别逻辑（示例：使用OpenCV的车牌检测）
        img = self.image.copy()
        #一般情况下，新能源车牌H分量的值通常都在30-80之间
        self.hsvLower = np.array([35,60,60])
        self.hsvUpper = np.array([85,255,255])    
        
        #plt.imshow(imgCarPlate[:,:,::-1])
        img4locate = self.imagePreProcess(img)
        target = self.locate_plate(img4locate, img)
        (x0,y0,x1,y1) = target

        cv2.rectangle(img, (x0,y0), (x1,y1), (0,255,0), 2)

        self.display_image(img)  # 更新显示识别后的图像